﻿using gov.va.med.vbecs.Common;
using System;
using System.ComponentModel;
using System.Windows.Forms;

namespace gov.va.med.vbecs.GUI.controls
{
    /// <summary>
    /// Base class to be used in WinForm and WPF PatientInfoBar
    /// </summary>
    public class PatientInfoToolbarBase : NotifyPropertyChanged
    {
        /// <summary>
        /// Display Medication Tooltip
        /// </summary>
        public const string DISPLAY_MEDICATION_TOOLTIP = "Display Medication Profile.";
        /// <summary>
        /// Display Blood Availability Tooltip
        /// </summary>
        public const string DISPLAY_BLOOD_AVAIL_TOOLTIP = "Display Patient Blood Availability.";
        /// <summary>
        /// Display Special Instructions and Transfusion Requirements Tooltip
        /// </summary>
        public const string DISPLAY_SI_AND_TR_TOOLTIP = "Display Special Instructions and Transfusion Requirements.";
        /// <summary>
        ///  Display Transfusion Reaction History Tooltip
        /// </summary>
        public const string DISPLAY_TRANSFUSION_HISTORY_TOOLTIP = "Display Transfusion Reaction History.";
        /// <summary>
        /// Display Recent Transfusion/Issue History tooltip
        /// </summary>
        public const string DISPLAY_RECENT_TRANSFUSION_ISSUE_TOOLTIP = "Display Recent Transfusion/Issue History.";
        /// <summary>
        /// Display recent orders Tooltip
        /// </summary>
        public const string DISPLAY_RECENT_ORDERS_TOOLTIP = "Display recent orders.";
        
        /// <summary>
        /// Constructor 
        /// </summary>
        public PatientInfoToolbarBase()
        {
            this.singleRowDisplay = true;
            //UserActivityEvent = new EventHandler(OnUserActivityEvent);
        }

        /// <summary>
        /// Constructor 
        /// </summary>
        public PatientInfoToolbarBase(BOL.Patient patient)
            :this()
        {
            Patient = patient;
        }

        #region Control Events
        /// <summary>
        /// Event needed for locking - signals to the containing form that the user is "doin' stuff"
        /// </summary>
        protected void OnUserActivity()
        {
            try
            {
                if (this.UserActivity != null)
                    this.UserActivity(this, System.EventArgs.Empty);
            }
            catch (ObjectDisposedException) {/*swallow the ObjectDisposedException ocurring on a lock expiration*/}
        }

        /// <summary>
        /// Bubbles up the user activity to the container form
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BubbleUserActivity(object sender, System.EventArgs e)
        {
            this.OnUserActivity();
        }

        /// <summary>
        /// Calls the event when special instructions and patient transfusion requirements button is being clicked
        /// </summary>
        /// <param name="parentForm"></param>
        public void OnSpecialInstructionsClick(VbecsBaseForm parentForm)
        {
            this.OnUserActivity();

            if (this._patient != null)
            {
                DlgDisplayTransReqsAndSpecIns dlg = new DlgDisplayTransReqsAndSpecIns(this._patient);

                dlg.RefreshParentLockTimeEvent += new EventHandler(this.BubbleUserActivity);

                if (dlg.DialogResult != DialogResult.Abort)
                {
                    //VbecsBaseForm parentForm = (VbecsBaseForm)this.FindForm();
                    dlg.ShowDialogFromLockedParent(parentForm, false);
                }
                else
                    dlg.Dispose();
            }
        }

        /// <summary>
        /// Calls the event when blood availability button is being clicked
        /// </summary>
        /// <param name="parentForm"></param>
        public void OnBloodAvailabilityClick(VbecsBaseForm parentForm)
        {
            this.OnUserActivity();

            if (this._patient != null)
            {
                DlgReportBloodAvailability dlg = new DlgReportBloodAvailability(_bloodAvailability);
                dlg.RefreshParentLockTimeEvent += new EventHandler(this.BubbleUserActivity);
                //VbecsBaseForm parentForm = (VbecsBaseForm)this.FindForm();
                dlg.ShowDialogFromLockedParent(parentForm, false);
            }
        }

        /// <summary>
        /// Calls the event when patient VBECS data button is being clicked
        /// </summary>
        /// <param name="parentForm"></param>
        public void OnRecentOrdersClick(VbecsBaseForm parentForm)
        {
            this.OnUserActivity();

            if (this._patient == null)
                return;

            DlgRecentOrders dlg = new DlgRecentOrders(this._patient);
            dlg.RefreshParentLockTimeEvent += new EventHandler(this.BubbleUserActivity);
            //VbecsBaseForm parentForm = (VbecsBaseForm)this.FindForm();
            dlg.ShowDialogFromLockedParent(parentForm, false);
        }

        /// <summary>
        /// Calls the event when patient current medications button is being clicked
        /// </summary>
        /// <param name="parentForm"></param>
        public void OnCurrentMedsClick(VbecsBaseForm parentForm)
        {
            this.OnUserActivity();

            if (this._patient == null)
                return;

            FrmMedProfile dlg = new FrmMedProfile(this._patient);
            dlg.RefreshParentLockTimeEvent += new EventHandler(this.BubbleUserActivity);
            //VbecsBaseForm parentForm = (VbecsBaseForm)this.FindForm();
            dlg.ShowDialogFromLockedParent(parentForm, false);
        }

        /// <summary>
        /// Calls the event when patient transfusion history button is being clicked
        /// </summary>
        /// <param name="parentForm"></param>
        public void OnRecentTransfusionsIssuesClick(VbecsBaseForm parentForm)
        {
            this.OnUserActivity();

            if (this._patient == null)
                return;

            DlgRecentTransfusionsIssues dlg = new DlgRecentTransfusionsIssues(this._patient);
            dlg.RefreshParentLockTimeEvent += new EventHandler(this.BubbleUserActivity);
            //VbecsBaseForm parentForm = (VbecsBaseForm)this.FindForm();
            dlg.ShowDialogFromLockedParent(parentForm, false);
        }

        /// <summary>
        /// Calls the event when patient clinical data button is being clicked
        /// </summary>
        /// <param name="parentForm"></param>
        public void OnTransfusionReactionHistoryClick(VbecsBaseForm parentForm)
        {
            this.OnUserActivity();

            if (this._patient == null)
                return;

            DlgTransfusionReactionHistory dlg = new DlgTransfusionReactionHistory(this._patient);
            dlg.RefreshParentLockTimeEvent += new EventHandler(this.BubbleUserActivity);
            //VbecsBaseForm parentForm = (VbecsBaseForm)this.FindForm();
            dlg.ShowDialogFromLockedParent(parentForm, false);
        }


        #endregion

        #region Properties
        private BOL.BloodAvailability _bloodAvailability;

        /// <summary>
        /// Location of Help file
        /// </summary>
        public string HelpNamespace { get { return Environment.CurrentDirectory + "\\VBECS.chm"; } }

        /// <summary>
        /// Event needed for locking - signals to the containing form that the user is "doin' stuff"
        /// </summary>
        public event EventHandler UserActivity;

        /// <summary>
        /// Property name to be used in PropertyChanged event so lambda expression doesn't have to be used (which is slow)
        /// </summary>
        public string ParentFormInstanceGuidPropName = "ParentFormInstanceGuid";
        private System.Guid parentFormInstanceGuid;
        /// <summary>
        /// The Guid of the container holing the toolbar (where any locks may have been created)
        /// Need this for child-form closing
        /// </summary> 
        public System.Guid ParentFormInstanceGuid
        {
            get
            {
                return parentFormInstanceGuid;
            }
            set
            {
                parentFormInstanceGuid = value;
                RaisePropertyChanged(() => ParentFormInstanceGuid);
            }
        }

        /// <summary>
        /// Property name to be used in PropertyChanged event so lambda expression doesn't have to be used (which is slow)
        /// </summary>
        public string SingleRowDisplayPropName = "SingleRowDisplay";
        private bool singleRowDisplay;
        /// <summary>
        /// Display tool bar as single or double row
        /// </summary>
        public bool SingleRowDisplay
        {
            get
            {
                return singleRowDisplay;
            }
            set
            {
                singleRowDisplay = value;
                RaisePropertyChanged(() => SingleRowDisplay);
            }
        }

        /// <summary>
        /// Property name to be used in PropertyChanged event so lambda expression doesn't have to be used (which is slow)
        /// </summary>
        public string EnabledSIAndTRButtonPropName = "EnabledSIAndTRButton";
        private bool enabledSIAndTRButton;
        /// <summary>
        /// Indicates whether the SI (and) TR button is enabled
        /// </summary>
        public bool EnabledSIAndTRButton
        {
            get
            {
                return enabledSIAndTRButton;
            }
            set
            {
                // only beep one time!
                if (value && !enabledSIAndTRButton) Common.Utility.MessageBeep();
                enabledSIAndTRButton = value;
                RaisePropertyChanged(() => EnabledSIAndTRButton);
            }
        }

        /// <summary>
        /// Property name to be used in PropertyChanged event so lambda expression doesn't have to be used (which is slow)
        /// </summary>
        public string EnabledBloodAvailButtonPropName = "EnabledBloodAvailButton";
        private bool enabledBloodAvailButton;
        /// <summary>
        /// Indicates whether the blood availability button is enabled
        /// </summary>
        public bool EnabledBloodAvailButton
        {
            get
            {
                return enabledBloodAvailButton;
            }
            set
            {
                enabledBloodAvailButton = value;
                RaisePropertyChanged(() => EnabledBloodAvailButton);
            }
        }

        /// <summary>
        /// Property name to be used in PropertyChanged event so lambda expression doesn't have to be used (which is slow)
        /// </summary>
        public string EnabledRecentOrdersButtonPropName = "EnabledRecentOrdersButton";
        private bool enabledRecentOrdersButton;
        /// <summary>
        /// Indicates whether the Recent Orders button is enabled
        /// </summary>
        public bool EnabledRecentOrdersButton
        {
            get
            {
                return enabledRecentOrdersButton;
            }
            set
            {
                enabledRecentOrdersButton = value;
                RaisePropertyChanged(() => EnabledRecentOrdersButton);
            }
        }

        /// <summary>
        /// Property name to be used in PropertyChanged event so lambda expression doesn't have to be used (which is slow)
        /// </summary>
        public string EnabledMedicationButtonPropName = "EnabledMedicationButton";
        private bool enabledMedicationButton;
        /// <summary>
        /// Indicates whether the Medication Profile button is enabled
        /// </summary>
        public bool EnabledMedicationButton
        {
            get
            {
                return enabledMedicationButton;
            }
            set
            {
                enabledMedicationButton = value;
                RaisePropertyChanged(() => EnabledMedicationButton);
            }
        }

        /// <summary>
        /// Property name to be used in PropertyChanged event so lambda expression doesn't have to be used (which is slow)
        /// </summary>
        public string EnabledRecentTransfusionsIssuesButtonPropName = "EnabledRecentTransfusionsIssuesButton";
        private bool enabledRecentTransfusionsIssuesButton;
        /// <summary>
        /// Indicates whether the Recent Transfusios/Issued Units button is enabled
        /// </summary>
        public bool EnabledRecentTransfusionsIssuesButton
        {
            get
            {
                return enabledRecentTransfusionsIssuesButton;
            }
            set
            {
                enabledRecentTransfusionsIssuesButton = value;
                RaisePropertyChanged(() => EnabledRecentTransfusionsIssuesButton);
            }
        }

        /// <summary>
        /// Property name to be used in PropertyChanged event so lambda expression doesn't have to be used (which is slow)
        /// </summary>
        public string EnabledTransfusionReactionHistoryButtonPropName = "EnabledTransfusionReactionHistoryButton";
        private bool enabledTransfusionReactionHistoryButton;
        /// <summary>
        /// Indicates whether the Transfusion Reaction History button is enabled
        /// </summary>
        public bool EnabledTransfusionReactionHistoryButton
        {
            get
            {
                return enabledTransfusionReactionHistoryButton;
            }
            set
            {
                enabledTransfusionReactionHistoryButton = value;
                RaisePropertyChanged(() => EnabledTransfusionReactionHistoryButton);
            }
        }

        /// <summary>
        /// Property name to be used in PropertyChanged event so lambda expression doesn't have to be used (which is slow)
        /// </summary>
        public string PatientPropName = "Patient";
        private BOL.Patient _patient;
        /// <summary>
        /// Current working Patient object
        /// </summary>
        public BOL.Patient Patient
        {
            get
            {
                return this._patient;
            }
            set
            {
                _patient = value;
                SetPatient(_patient);
                RaisePropertyChanged(() => Patient);
            }
        }
        #endregion

        #region Methods

        private void SetPatient(BOL.Patient patient)
        {
            // reset button
            this.EnabledSIAndTRButton = false;

            if (patient != null)
            {
                //Medication button is always enabled
                EnabledMedicationButton = true;
                //
                // BR_16.01 - Enable only when there are Special Instructions or Transfusion Requirements to display
                this.EnabledSIAndTRButton = (BOL.PatientSpecialInstruction.DoActiveSpecialInstructionsExistForPatient(patient.PatientGuid) ||
                    BOL.PatientTransfusionRequirement.DoTransfusionRequirementsExistForPatient(patient.PatientGuid));
                //
                //Adding the patient filter to UC_49 report
                this._bloodAvailability = new BOL.BloodAvailability();
                this._bloodAvailability.IsIndividualPatientReport = true;
                this._bloodAvailability.IsMultiDivisional = false;
                this._bloodAvailability.PatientGuid = patient.PatientGuid;
                //Find the number of blood units for selected patient. Enable only if entries found
                int count = _bloodAvailability.GetBloodAvailabilityCount();
                EnabledBloodAvailButton = (count > 0);
                //
                //Find the number of patient transfusion reactions. Enable only if entries found
                int transfusionReactions = BOL.PatientTransfusionReaction.GetTransfusionReactionsForPatientCount(patient.PatientGuid);
                EnabledTransfusionReactionHistoryButton = (transfusionReactions > 0);
                //
                //Find the number of recent transfusions/issues. Enable only if entries found
                int issueTransfusions = BOL.PatientTransfusion.GetRecentlyIssuedAndTransfusedUnitsForPatientCount(patient.PatientGuid);
                EnabledRecentTransfusionsIssuesButton = (issueTransfusions > 0);
                //
                //Find the number of recent test orders. Enable only if entries found
                int recentTestOrders = BOL.OrderedTest.GetRecentTestOrdersCount(patient.PatientGuid);
                if (recentTestOrders > 0)
                {
                    EnabledRecentOrdersButton = true;
                }
                else
                {
                    //No need to check component orders if we've already found test orders.
                    int recentComponentOrders = BOL.OrderedComponent.GetRecentComponentOrdersCount(patient.PatientGuid);
                    if (recentComponentOrders > 0)
                    {
                        EnabledRecentOrdersButton = true;
                    }
                    else
                    {
                        EnabledRecentOrdersButton = false;
                    }
                }
            }
            else
            {
                this.EnableAllButtons(false);
            }
        }

        /// <summary>
        /// EnableAllButtons
        /// </summary>
        /// <param name="enable"></param>
        public void EnableAllButtons(bool enable)
        {
            EnabledBloodAvailButton = enable;
            EnabledRecentOrdersButton = enable;
            EnabledMedicationButton = enable;
            EnabledTransfusionReactionHistoryButton = enable;
            EnabledSIAndTRButton = enable;
            EnabledRecentTransfusionsIssuesButton = enable;
        }

        #endregion

    }
}
